/*
 * Routines for handling product info
 */

#include "libfma.h"
#include "lf_stdio.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_product_def.h"

/*
 * Local structs
 */
struct lf_product_info_alias {
  char *alias;
  char *product_id;
  struct lf_product_info_alias *next;	/* alias cache */
};

/*
 * local static data
 */
static struct {
  char *install;	/* FMS install dir */

  struct lf_enclosure_def *ed_cache;	/* enclosure data */
  struct lf_linecard_def *ld_cache;	/* linecard data */
  struct lf_nic_def *nd_cache;		/* nic data */
  struct lf_product_info_alias *alias_cache;	/* alias cache */
} L;

/*
 * local routines
 */
static int set_info_dirs(void);
static FILE *open_product_info(char *product_info);
static void lf_read_product_aliases(void);
static struct lf_enclosure_def *read_enclosure_definition(char *product_id);
static int read_enclosure_defs(FILE *fp, struct lf_enclosure_def *dp);
static int make_enclosure_allocations(struct lf_enclosure_def *dp);
static int read_enclosure_conns(FILE *fp, struct lf_enclosure_def *dp);

static struct lf_linecard_def *read_linecard_definition(char *product_id);
static int read_linecard_defs(FILE *fp, struct lf_linecard_def *dp);
static int make_linecard_allocations(struct lf_linecard_def *dp);
static int read_linecard_pin_xbar_conns(FILE *fp, struct lf_linecard_def *dp);
static int read_linecard_pin_xcvr_conns(FILE *fp, struct lf_linecard_def *dp);
static int read_linecard_xbar_xcvr_conns(FILE *fp, struct lf_linecard_def *dp);
static int read_linecard_xcvr_labels(FILE *fp, struct lf_linecard_def *dp);

static struct lf_nic_def *read_nic_definition(char *product_id);
static int read_nic_defs(FILE *fp, struct lf_nic_def *dp);

/*
 * initialize product info subsystem
 */
int
lf_init_product_info(
  char *install)
{
  if (install != NULL) {
    LF_DUP_STRING(L.install, install);	/* save FMS install directory */
  }
  return 0;

 except:
  return -1;
}

/*
 * See if there is an alias for a product ID string.  Return the string
 * itself if no alias found.
 */
char *
lf_product_id_alias(
  char *alias)
{
  struct lf_product_info_alias *piap;

  /* if no aliases yet, read the file */
  if (L.alias_cache == NULL) {
    lf_read_product_aliases();
  }

  /* search cached aliases */
  piap = L.alias_cache;
  while (piap != NULL) {
    if (strcmp(piap->alias, alias) == 0) {
      return piap->product_id;
    }
    piap = piap->next;
  }

  return alias;
}

/*
 * Read the aliases file and cache results
 */
static void
lf_read_product_aliases()
{
  FILE *fp;
  struct lf_product_info_alias *piap;
  lf_string_t buf;
  char *wp[LF_STRING_LEN];
  int wc;

  fp = open_product_info("aliases");
  if (fp == NULL) return;

  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {
    if (wc == 2) {
      LF_CALLOC(piap, struct lf_product_info_alias, 1);
      LF_DUP_STRING(piap->alias, wp[0]);
      LF_DUP_STRING(piap->product_id, wp[1]);
      piap->next = L.alias_cache;
      L.alias_cache = piap;
    }
  }

 except:
  fclose(fp);
  return;
}

/*
 * return product information for an enclosure
 */
struct lf_enclosure_def *
lf_get_enclosure_definition(
  char *product_id)
{
  struct lf_enclosure_def *dp;

  /* first, search cache for the info */
  dp = L.ed_cache;
  while (dp != NULL) {
    if (strcmp(dp->product_id, product_id) == 0) {
      return dp;
    }
    dp = dp->next;
  }

  /* not in cache, read it from disk */
  dp = read_enclosure_definition(product_id);

  /* if we got it, save it in cache */
  if (dp != NULL) {
    dp->next = L.ed_cache;
    L.ed_cache = dp;
  }

  return dp;		/* return what we have */
}

/*
 * Make sure the directory settings are valid
 */
static int
set_info_dirs()
{
  char *path;
  static char *path_buf;
  static char *path_elem[LF_STRING_LEN];
  static int npath;
  lf_string_t fullname;
  char *p;
  FILE *fp;
  int i;

  /* if unset, check environment */
  if (L.install == NULL) {
    L.install = getenv(LF_ENV_FMS_INSTALL);
  }

  /* Still NULL - search through PATH and find the dir */
  if (L.install == NULL) {

    /* Get and parse path */
    path = getenv("PATH");
    if (path == NULL) {
      LF_ERROR(("Unable to get PATH from environment"));
    }
    LF_DUP_STRING(path_buf, path);	/* local copy */

    line2words_nullok(path_buf, path_elem, ":", 0, &npath);

    /* Try opening "productinfo/aliases" at each location in PATH */
    for (i=0; i<npath; ++i) {

      /* Trim last dir from each path entry */
      p = strrchr(path_elem[i], '/');
      if (p == NULL) {
	p = path_elem[i];
      }
      *p = '\0';

      /* Prepend PATH and open unless empty */
      if (path_elem[i][0] != '\0') {
	sprintf(fullname, "%s/productinfo/aliases", path_elem[i]);
	fp = fopen(fullname, "r");
      } else {
	fp = fopen("productinfo/aliases", "r");
      }

      /* If no error, this is it! */
      if (fp != NULL) {
	fclose(fp);
	if (path_elem[i][0] != '\0') {
	  LF_DUP_STRING(L.install, path_elem[i]);
	} else {
	  LF_DUP_STRING(L.install, ".");
	}
	break;
      }
    }
  }

  if (L.install == NULL) {
    LF_ERROR(("Cannot locate productinfo dir!"));
  }
  return 0;

 except:
  return -1;
}

/*
 * Open file containing product info
 */
static FILE *
open_product_info(
  char *product_id)
{
  lf_string_t name;
  FILE *fp;
  int rc;

  /* Set/find productinfo directory */
  rc = set_info_dirs();
  if (rc == -1) {
    return NULL;
  }

  sprintf(name, "%s/%s/%s", L.install, LF_DFLT_INFO_DIR, product_id);
  fp = fopen(name, "r");
  if (fp == NULL) {
    LF_ERROR(("Cannot open product id file at %s", name));
  }
  return fp;

 except:
  return NULL;
}
/*
 * Read enclosure definition from a file
 */
static struct lf_enclosure_def *
read_enclosure_definition(
  char *product_id)
{
  FILE *fp;
  struct lf_enclosure_def *dp;
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int wc;
  int rc;

  /* init for cleanup */
  fp = NULL;
  dp = NULL;

  /* convert to alias if there is one */
  product_id = lf_product_id_alias(product_id);

  /* allocate and create product file name */
  fp = open_product_info(product_id);
  if (fp == NULL) {
    LF_ERROR(("Cannot open product id file for %s", product_id));
  }

  /* allocate a struct for saving the definition */
  LF_CALLOC(dp, struct lf_enclosure_def, 1);
  LF_DUP_STRING(dp->product_id, product_id);

  /* parse the file */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    /* read info about how many slots, etc. */
    if (strcmp(wp[0], "definitions") == 0) {
      rc = read_enclosure_defs(fp, dp);
      if (rc == -1) LF_ERROR(("reading generic enclosure defs"));

      /* make enclosure allocations based on definitions */
      rc = make_enclosure_allocations(dp);
      if (rc == -1) {
	LF_ERROR(("making enclosure allocations for %s", dp->product_id));
      }

    /* read connection info */
    } else if (strcmp(wp[0], "connections") == 0) {
      rc = read_enclosure_conns(fp, dp);
      if (rc == -1) LF_ERROR(("reading generic enclosure connections"));
      
    } else {
      LF_ERROR(("Unexpected keyword \"%s\" while reading %s\n",
	         wp[0], dp->product_id));
    }
  }
  fclose(fp);
  return dp;

 except:
  if (fp != NULL) fclose(fp);
  LF_FREE(dp);
  return NULL;
}

/*
 * make enclosure allocations
 */
static int
make_enclosure_allocations(
  struct lf_enclosure_def *dp)
{
  int slot;

  /* total number of slots */
  dp->num_slots = dp->num_lc_slots + dp->num_bp_slots;

  /* allocate slot connections */
  LF_CALLOC(dp->conn, struct lf_slot_conn *, dp->num_slots);

  for (slot=0; slot<dp->num_slots; ++slot) {
    LF_CALLOC(dp->conn[slot], struct lf_slot_conn, dp->num_pins);
  }
  return 0;

 except:
  return -1;
}

/*
 * read generic definitions for an enclosure
 */
static int
read_enclosure_defs(
  FILE *fp,
  struct lf_enclosure_def *dp)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int wc;

  /* parse the file */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    if (strcmp(wp[0], "end_definitions") == 0) {
      return 0;

    } else if (strcmp(wp[0], "slots") == 0) {
      dp->num_lc_slots = atoi(wp[1]);

    } else if (strcmp(wp[0], "backplane_slots") == 0) {
      dp->num_bp_slots = atoi(wp[1]);

    } else if (strcmp(wp[0], "pins") == 0) {
      dp->num_pins = atoi(wp[1]);

    } else if (strcmp(wp[0], "slotbase") == 0) {
      dp->lc_slotbase = atoi(wp[1]);

    } else if (strcmp(wp[0], "backplane_slotbase") == 0) {
      dp->bp_slotbase = atoi(wp[1]);

    } else if (strcmp(wp[0], "backplane_product_id") == 0) {
      LF_DUP_STRING(dp->bp_product_id, wp[1]);

    } else {
      LF_ERROR(("Unrecognized keyword \"%s\" parsing product info for %s",
	         wp[0], dp->product_id));
    }

  }

  LF_ERROR(("Unexpected EOF parsing product info for %s", dp->product_id));
 except:
  return -1;
}

/*
 * read conneciton info for an enclosure
 */
static int
read_enclosure_conns(
  FILE *fp,
  struct lf_enclosure_def *dp)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int wc;

  /* parse the file */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    if (strcmp(wp[0], "end_connections") == 0) {
      return 0;

    } else if (wc == 4) {
      int slot1, pin1;
      int slot2, pin2;
      
      slot1 = atoi(wp[0]);
      pin1 = atoi(wp[1]);
      slot2 = atoi(wp[2]);
      pin2 = atoi(wp[3]);

      dp->conn[slot1][pin1].slot = slot2;
      dp->conn[slot1][pin1].pin = pin2;
      dp->conn[slot2][pin2].slot = slot1;
      dp->conn[slot2][pin2].pin = pin1;

    } else {
      LF_ERROR(("Unrecognized keyword parsing connecitons for %s",
	         wp[0], dp->product_id));
    }
  }

  LF_ERROR(("unexpected EOF reading connections for %s", dp->product_id));
 except:
  return -1;
}

/*
 * return product information for a linecard
 */
struct lf_linecard_def *
lf_get_linecard_definition(
  char *product_id)
{
  struct lf_linecard_def *dp;

  /* first, search cache for the info */
  dp = L.ld_cache;
  while (dp != NULL) {
    if (strcmp(dp->product_id, product_id) == 0) {
      return dp;
    }
    dp = dp->next;
  }

  /* not in cache, read it from disk */
  dp = read_linecard_definition(product_id);

  /* if we got it, save it in cache */
  if (dp != NULL) {
    dp->next = L.ld_cache;
    L.ld_cache = dp;
  }

  return dp;		/* return what we have */
}

/*
 * read a linecard file
 */
static struct lf_linecard_def *
read_linecard_definition(
  char *product_id)
{
  FILE *fp;
  char *name;
  struct lf_linecard_def *dp;
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int wc;
  int rc;

  /* init for cleanup */
  fp = NULL;
  name = NULL;
  dp = NULL;

  /* convert to alias if there is one */
  product_id = lf_product_id_alias(product_id);

  /* open file for product info */
  fp = open_product_info(product_id);
  if (fp == NULL) {
    LF_ERROR(("Cannot open product id file for %s", product_id));
  }

  /* allocate a struct for saving the definition */
  LF_CALLOC(dp, struct lf_linecard_def, 1);
  LF_DUP_STRING(dp->product_id, product_id);

  /* parse the file */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    /* read info about how many slots, etc. */
    if (strcmp(wp[0], "definitions") == 0) {
      rc = read_linecard_defs(fp, dp);
      if (rc == -1) LF_ERROR(("reading generic linecard defs"));

      /* make linecard allocations based on definitions */
      rc = make_linecard_allocations(dp);
      if (rc == -1) {
	LF_ERROR(("making linecard allocations for %s", dp->product_id));
      }

    /* read xcvr label info */
    } else if (strcmp(wp[0], "xcvr_labels") == 0) {
      rc = read_linecard_xcvr_labels(fp, dp);
      if (rc == -1) {
	LF_ERROR(("reading xcvr labels for %s", dp->product_id));
      }

    /* read xbar-xcvr connection info */
    } else if (strcmp(wp[0], "xbar_xcvr_links") == 0) {
      rc = read_linecard_xbar_xcvr_conns(fp, dp);
      if (rc == -1) {
	LF_ERROR(("reading xbar-xvcr connections for %s", dp->product_id));
      }

    /* read xbar-pin connection info */
    } else if (strcmp(wp[0], "pin_xbar_links") == 0) {
      rc = read_linecard_pin_xbar_conns(fp, dp);
      if (rc == -1) {
	LF_ERROR(("reading xbar-pin connections for %s", dp->product_id));
      }

    /* read xcvr-pin connection info */
    } else if (strcmp(wp[0], "pin_xcvr_links") == 0) {
      rc = read_linecard_pin_xcvr_conns(fp, dp);
      if (rc == -1) {
	LF_ERROR(("reading xcvr-pin connections for %s", dp->product_id));
      }

    } else {
      LF_ERROR(("Unexpected keyword \"%s\" while reading %s\n",
	         wp[0], dp->product_id));
    }
  }
  fclose(fp);
  return dp;

 except:
  if (fp != NULL) fclose(fp);
  LF_FREE(dp);
  return NULL;
}

/*
 * read generic definitions for a linecard
 */
static int
read_linecard_defs(
  FILE *fp,
  struct lf_linecard_def *dp)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int wc;

  /* defaults */
  dp->xbar_id_style = LF_XS_NONE;
  dp->lc_type = LF_LC_TYPE_UNKNOWN;

  /* parse the file */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    if (strcmp(wp[0], "end_definitions") == 0) {
      goto all_done;

    } else if (strcmp(wp[0], "pins") == 0) {
      dp->num_pins = atoi(wp[1]);

    } else if (strcmp(wp[0], "xbars") == 0) {
      dp->num_xbars = atoi(wp[1]);

    } else if (strcmp(wp[0], "xbar_ports") == 0) {
      dp->num_xbar_ports = atoi(wp[1]);

    } else if (strcmp(wp[0], "transceivers") == 0) {
      dp->num_xcvrs = atoi(wp[1]);

    } else if (strcmp(wp[0], "transceiver_ports") == 0) {
      dp->num_xcvr_ports = atoi(wp[1]);

    } else if (strcmp(wp[0], "transceiver_string") == 0) {
      LF_DUP_STRING(dp->xcvr_port_string, wp[1]);

    } else if (strcmp(wp[0], "link_type") == 0) {
      LF_DUP_STRING(dp->link_type, wp[1]);

    } else if (strcmp(wp[0], "qd_low_bit") == 0) {
      dp->qd_low_bit = atoi(wp[1]);

    } else if (strcmp(wp[0], "qd_high_bit") == 0) {
      dp->qd_high_bit = atoi(wp[1]);

    } else if (strcmp(wp[0], "type") == 0) {
      if (strcmp(wp[1], "comm") == 0) {
	dp->lc_type = LF_LC_TYPE_COMM;
      } else if (strcmp(wp[1], "monitor") == 0) {
	dp->lc_type = LF_LC_TYPE_MONITOR;
      } else if (strcmp(wp[1], "mechanical") == 0) {
	dp->lc_type = LF_LC_TYPE_MECHANICAL;
      } else {
	LF_ERROR(("Unrecognized linecard type \"%s\" for %s",
	         wp[1], dp->product_id));
      }

    } else if (strcmp(wp[0], "xbar_id_style") == 0) {
      if (strcmp(wp[1], "serialx16") == 0) {
	dp->xbar_id_style = LF_XS_SERIALX16;
      } else if (strcmp(wp[1], "same") == 0) {
	dp->xbar_id_style = LF_XS_SAME;
      } else if (strcmp(wp[1], "none") == 0) {
	dp->xbar_id_style = LF_XS_NONE;
      } else {
	LF_ERROR(("Unrecognized xbar_id_style \"%s\" for %s",
	         wp[1], dp->product_id));
      }

    } else {
      LF_ERROR(("Unrecognized keyword \"%s\" parsing product info for %s",
	         wp[0], dp->product_id));
    }
  }

  LF_ERROR(("Unexpected EOF parsing product info for %s", dp->product_id));

 all_done:
  if (dp->lc_type == LF_LC_TYPE_UNKNOWN) {
    LF_ERROR(("Missing linecard type for %s", dp->product_id));
  }
  return 0;

 except:
  return -1;
}

/*
 * read connection info between xbars and xcvrs
 */
static int
read_linecard_xbar_xcvr_conns(
  FILE *fp,
  struct lf_linecard_def *dp)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int wc;

  /* parse the file */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    if (strcmp(wp[0], "end_xbar_xcvr_links") == 0) {
      return 0;

    } else if (wc == 4) {
      int xbar_no;
      int port;
      struct lf_linecard_xbar_conn *cp;
      
      xbar_no = atoi(wp[0]);
      port = atoi(wp[1]);

      /* allocate and fill in the link */
      LF_CALLOC(cp, struct lf_linecard_xbar_conn, 1);
      cp->index = atoi(wp[2]);
      cp->port = atoi(wp[3]);
      dp->xbar_conn[xbar_no][port] = cp;

    } else {
      LF_ERROR(("Unrecognized keyword parsing xbar-xcvr for %s",
	         wp[0], dp->product_id));
    }
  }

  LF_ERROR(("unexpected EOF reading xbar-xcvr for %s", dp->product_id));

 except:
  return -1;
}

/*
 * read connection info between linecard pins and xbars
 */
static int
read_linecard_pin_xbar_conns(
  FILE *fp,
  struct lf_linecard_def *dp)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int wc;

  /* parse the file */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    if (strcmp(wp[0], "end_pin_xbar_links") == 0) {
      return 0;

    } else if (wc == 3) {
      int pin;
      
      pin = atoi(wp[0]);

      dp->pin_conn[pin].is_xbar = TRUE;
      dp->pin_conn[pin].index = atoi(wp[1]);
      dp->pin_conn[pin].port = atoi(wp[2]);

    } else {
      LF_ERROR(("Unrecognized keyword \"%s\" parsing pin-xbar for %s",
	         wp[0], dp->product_id));
    }
  }

  LF_ERROR(("unexpected EOF reading pin-xbar for %s", dp->product_id));
 except:
  return -1;
}

/*
 * read connection info between linecard pins and xcvrs
 */
static int
read_linecard_pin_xcvr_conns(
  FILE *fp,
  struct lf_linecard_def *dp)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int wc;

  /* parse the file */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    if (strcmp(wp[0], "end_pin_xcvr_links") == 0) {
      return 0;

    } else if (wc == 3) {
      int pin;
      
      pin = atoi(wp[0]);

      dp->pin_conn[pin].is_xbar = FALSE;
      dp->pin_conn[pin].index = atoi(wp[1]);
      dp->pin_conn[pin].port = atoi(wp[2]);

    } else {
      LF_ERROR(("Unrecognized keyword \"%s\" parsing pin-xcvr for %s",
	         wp[0], dp->product_id));
    }
  }

  LF_ERROR(("unexpected EOF reading pin-xcvr for %s", dp->product_id));
 except:
  return -1;
}

/*
 * read transceiver labels for a linecard
 */
static int
read_linecard_xcvr_labels(
  FILE *fp,
  struct lf_linecard_def *dp)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int wc;

  /* parse the file */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    if (strcmp(wp[0], "end_xcvr_labels") == 0) {
      return 0;

    } else if (wc == 2) {
      
      dp->xcvr_labels[atoi(wp[0])] = atoi(wp[1]);

    } else {
      LF_ERROR(("Unrecognized keyword parsing xcvr labels for %s",
	         wp[0], dp->product_id));
    }
  }

  LF_ERROR(("unexpected EOF reading pin-xcvr for %s", dp->product_id));
 except:
  return -1;
}

/*
 * make linecard allocations
 */
static int
make_linecard_allocations(
  struct lf_linecard_def *dp)
{
  int xbar_no;

  /* allocate holders for pin connections */
  LF_CALLOC(dp->pin_conn, struct lf_linecard_pin_conn, dp->num_pins);

  /* allocate pointers for xbar-xcvr links */
  LF_CALLOC(dp->xbar_conn, struct lf_linecard_xbar_conn **, dp->num_xbars);
  for (xbar_no=0; xbar_no < dp->num_xbars; ++xbar_no) {
    LF_CALLOC(dp->xbar_conn[xbar_no],
	      struct lf_linecard_xbar_conn *,
	      dp->num_xbar_ports);
  }

  /* labels for xcvrs */
  if (dp->num_xcvrs > 0) {
    LF_CALLOC(dp->xcvr_labels, int, dp->num_xcvrs);
  }
  return 0;

 except:
  return -1;
}

/*
 * return product information for an NIC
 */
struct lf_nic_def *
lf_get_nic_definition(
  char *product_id)
{
  struct lf_nic_def *dp;

  /* convert to alias if there is one */
  product_id = lf_product_id_alias(product_id);

  /* first, search cache for the info */
  dp = L.nd_cache;
  while (dp != NULL) {
    if (strcmp(dp->product_id, product_id) == 0) {
      return dp;
    }
    dp = dp->next;
  }

  /* not in cache, read it from disk */
  dp = read_nic_definition(product_id);

  /* if we got it, save it in cache */
  if (dp != NULL) {
    dp->next = L.nd_cache;
    L.nd_cache = dp;
  }

  return dp;		/* return what we have */
}

/*
 * Read nic definition from a file
 */
static struct lf_nic_def *
read_nic_definition(
  char *product_id)
{
  FILE *fp;
  struct lf_nic_def *dp;
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int wc;
  int rc;

  /* init for cleanup */
  fp = NULL;
  dp = NULL;

  /* allocate and create product file name */
  fp = open_product_info(product_id);
  if (fp == NULL) {
    LF_ERROR(("Cannot open product id file for %s", product_id));
  }

  /* allocate a struct for saving the definition */
  LF_CALLOC(dp, struct lf_nic_def, 1);
  LF_DUP_STRING(dp->product_id, product_id);

  /* parse the file */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    /* read info about how many slots, etc. */
    if (strcmp(wp[0], "definitions") == 0) {
      rc = read_nic_defs(fp, dp);
      if (rc == -1) LF_ERROR(("reading generic nic defs"));
    }

  }
  fclose(fp);
  return dp;

 except:
  if (fp != NULL) fclose(fp);
  LF_FREE(dp);
  return NULL;
}

/*
 * read generic definitions for an NIC
 */
static int
read_nic_defs(
  FILE *fp,
  struct lf_nic_def *dp)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  int wc;

  /* parse the file */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    if (strcmp(wp[0], "end_definitions") == 0) {
      return 0;

    } else if (strcmp(wp[0], "interfaces") == 0) {
      dp->num_links = atoi(wp[1]);

    } else if (strcmp(wp[0], "transceivers") == 0) {
      dp->num_xcvrs = atoi(wp[1]);

    } else if (strcmp(wp[0], "transceiver_ports") == 0) {
      dp->xcvr_ports = atoi(wp[1]);

    } else if (strcmp(wp[0], "ram_size") == 0) {
      dp->ram_size = atoi(wp[1]);

    } else {
      LF_ERROR(("Unrecognized keyword \"%s\" parsing product info for %s",
	         wp[0], dp->product_id));
    }

  }

  LF_ERROR(("Unexpected EOF parsing product info for %s", dp->product_id));
 except:
  return -1;
}
